<?php
/**
 * Склад → Инструменты → Ремонт
 * Файл: modules/warehouse/tool_repairs.php
 *
 * ВАЖНО: Возврат из ремонта идёт ТОЛЬКО туда же, откуда был сдан:
 * - from_employee_id -> to_employee_id
 * - else from_workshop_id -> to_workshop_id
 * - else from_location_id -> to_location_id
 */

require_once '../../config.php';
require_once '../../auth/guard.php';
require_once '../../lib/db.php';
require_once '../../lib/acl.php';

requirePermission('warehouse', 'view');

function h($s) { return htmlspecialchars((string)$s); }
function fmt_qty_local($v) {
    $s = (string)$v;
    if (strpos($s, '.') !== false) $s = rtrim(rtrim($s, '0'), '.');
    return $s === '' ? '0' : $s;
}

function log_action($userId, $action, $recordType, $recordId, $newValues) {
    $module = 'warehouse';
    $ip = db_escape($_SERVER['REMOTE_ADDR'] ?? '');
    $ua = db_escape($_SERVER['HTTP_USER_AGENT'] ?? '');
    $newJson = db_escape(json_encode($newValues, JSON_UNESCAPED_UNICODE));
    db_query("INSERT INTO user_logs (user_id, module, action, record_type, record_id, new_values, ip_address, user_agent)
              VALUES ($userId, '$module', '$action', '" . db_escape($recordType) . "', " . (int)$recordId . ", '$newJson', '$ip', '$ua')");
}

$userId = (int)($_SESSION['user_id'] ?? 0);

// локации (для формы старта)
$visibleLocations = ACL::getUserLocations($userId);
if (!$visibleLocations) {
    $visibleLocations = db_fetch_all("SELECT id, name FROM locations WHERE is_active=1 ORDER BY name");
}

// цеха (для формы старта)
$visibleWorkshops = ACL::getUserWorkshops($userId);
if (!$visibleWorkshops) {
    $visibleWorkshops = db_fetch_all("
        SELECT w.id, w.name, l.name AS location_name
        FROM workshops w
        JOIN locations l ON l.id=w.location_id
        WHERE w.is_active=1 AND l.is_active=1
        ORDER BY l.name, w.name
    ");
}

// сотрудники (для формы старта)
$employees = db_fetch_all("SELECT id, full_name FROM employees WHERE is_active=1 ORDER BY full_name");

// инструменты
$tools = db_fetch_all("
    SELECT i.id, i.name, i.unit
    FROM items i
    WHERE i.is_active=1 AND i.item_type='tool'
    ORDER BY i.name
");

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requirePermission('warehouse', 'edit');

    $mode = $_POST['mode'] ?? '';

    if ($mode === 'start') {
        $itemId = (int)($_POST['item_id'] ?? 0);
        $qty = trim($_POST['quantity'] ?? '');
        $sourceType = $_POST['source_type'] ?? 'location'; // location|workshop|employee
        $fromLocationId = (int)($_POST['from_location_id'] ?? 0);
        $fromWorkshopId = (int)($_POST['from_workshop_id'] ?? 0);
        $fromEmployeeId = (int)($_POST['from_employee_id'] ?? 0);
        $reason = trim($_POST['reason'] ?? '');
        $started = trim($_POST['started_date'] ?? date('Y-m-d'));
        $expected = trim($_POST['expected_date'] ?? '');

        if ($itemId <= 0) $errors[] = 'Выбери инструмент';
        if ($qty === '' || !is_numeric(str_replace(',', '.', $qty)) || (float)str_replace(',', '.', $qty) <= 0) $errors[] = 'Количество должно быть числом больше 0';
        if ($reason === '') $errors[] = 'Причина/комментарий обязателен';

        if ($sourceType === 'location') {
            if ($fromLocationId <= 0) $errors[] = 'Выбери локацию';
        } elseif ($sourceType === 'workshop') {
            if ($fromWorkshopId <= 0) $errors[] = 'Выбери цех';
        } elseif ($sourceType === 'employee') {
            if ($fromEmployeeId <= 0) $errors[] = 'Выбери сотрудника';
        } else {
            $errors[] = 'Неверный источник';
        }

        if (!$errors) {
            $qtyVal = (float)str_replace(',', '.', $qty);
            $qtySql = db_escape(number_format($qtyVal, 3, '.', ''));

            $fromLocSql = "NULL";
            $fromWsSql  = "NULL";
            $fromEmpSql = "NULL";

            if ($sourceType === 'location') $fromLocSql = (string)$fromLocationId;
            if ($sourceType === 'workshop') $fromWsSql  = (string)$fromWorkshopId;
            if ($sourceType === 'employee') $fromEmpSql = (string)$fromEmployeeId;

            $reasonSql = db_escape($reason);
            $startedSql = db_escape($started);
            $expectedSql = ($expected === '') ? 'NULL' : ("'" . db_escape($expected) . "'");

            db_query("INSERT INTO tool_repairs
                (item_id, from_location_id, from_workshop_id, from_employee_id, from_user_id, quantity, status, reason, started_date, expected_date, created_by)
                VALUES
                ($itemId, $fromLocSql, $fromWsSql, $fromEmpSql, $userId, '$qtySql', 'in_repair', '$reasonSql', '$startedSql', $expectedSql, $userId)
            ");

            $last = db_fetch_one("SELECT LAST_INSERT_ID() AS id");
            $rid = $last ? (int)$last['id'] : 0;

            log_action($userId, 'create', 'tool_repair', $rid, [
                'item_id' => $itemId,
                'quantity' => $qtyVal,
                'source_type' => $sourceType,
                'from_location_id' => ($sourceType==='location') ? $fromLocationId : null,
                'from_workshop_id' => ($sourceType==='workshop') ? $fromWorkshopId : null,
                'from_employee_id' => ($sourceType==='employee') ? $fromEmployeeId : null,
                'reason' => $reason,
                'started_date' => $started,
                'expected_date' => $expected
            ]);

            header('Location: tool_repairs.php?saved=1');
            exit;
        }

    } elseif ($mode === 'finish_return') {
        // Возврат в то же место, откуда сдали
        $repairId = (int)($_POST['repair_id'] ?? 0);
        $date = trim($_POST['finish_date'] ?? date('Y-m-d'));
        $comment = trim($_POST['comment'] ?? '');

        $rep = db_fetch_one("SELECT * FROM tool_repairs WHERE id=$repairId LIMIT 1");
        if (!$rep || $rep['status'] !== 'in_repair') $errors[] = 'Ремонт не найден или уже закрыт';

        if (!$errors) {
            $toLocSql = "NULL";
            $toWsSql  = "NULL";
            $toEmpSql = "NULL";
            $toType = null;

            if (!empty($rep['from_employee_id'])) {
                $toEmpSql = (string)(int)$rep['from_employee_id'];
                $toType = 'employee';
            } elseif (!empty($rep['from_workshop_id'])) {
                $toWsSql = (string)(int)$rep['from_workshop_id'];
                $toType = 'workshop';
            } elseif (!empty($rep['from_location_id'])) {
                $toLocSql = (string)(int)$rep['from_location_id'];
                $toType = 'location';
            } else {
                $errors[] = 'Неизвестно, откуда был сдан инструмент (нет from_location/workshop/employee)';
            }

            if (!$errors) {
                $qtySql = db_escape(number_format((float)$rep['quantity'], 3, '.', ''));
                $reasonSql = db_escape('Возврат с ремонта #' . $repairId . ($comment ? (': ' . $comment) : ''));

                db_query("INSERT INTO inventory_movements
                    (movement_type, item_type, item_id, quantity, to_location_id, to_workshop_id, to_employee_id, reason, reference_type, reference_id, movement_date, created_by)
                    VALUES
                    ('receipt', 'tool', " . (int)$rep['item_id'] . ", '$qtySql', $toLocSql, $toWsSql, $toEmpSql,
                     '$reasonSql', 'tool_repair', $repairId, '" . db_escape($date) . "', $userId)
                ");

                db_query("UPDATE tool_repairs
                          SET status='returned', finished_date='" . db_escape($date) . "'
                          WHERE id=$repairId");

                log_action($userId, 'update', 'tool_repair', $repairId, [
                    'status' => 'returned',
                    'finished_date' => $date,
                    'return_to' => $toType,
                    'comment' => $comment
                ]);

                header('Location: tool_repairs.php?saved=1');
                exit;
            }
        }

    } elseif ($mode === 'finish_writeoff') {
        // Списать в ремонте (причина обязательна)
        $repairId = (int)($_POST['repair_id'] ?? 0);
        $date = trim($_POST['finish_date'] ?? date('Y-m-d'));
        $comment = trim($_POST['comment'] ?? '');

        if ($comment === '') $errors[] = 'Причина списания обязательна';

        $rep = db_fetch_one("SELECT * FROM tool_repairs WHERE id=$repairId LIMIT 1");
        if (!$rep || $rep['status'] !== 'in_repair') $errors[] = 'Ремонт не найден или уже закрыт';

        if (!$errors) {
            $qtySql = db_escape(number_format((float)$rep['quantity'], 3, '.', ''));
            $reasonSql = db_escape('Списание в ремонте #' . $repairId . ': ' . $comment);

            // если известна локация источника — списываем с неё, иначе просто фиксируем списание без склада
            $fromLocSql = (!empty($rep['from_location_id'])) ? (string)(int)$rep['from_location_id'] : "NULL";

            db_query("INSERT INTO inventory_movements
                (movement_type, item_type, item_id, quantity, from_location_id, reason, reference_type, reference_id, movement_date, created_by)
                VALUES
                ('writeoff', 'tool', " . (int)$rep['item_id'] . ", '$qtySql', $fromLocSql,
                 '$reasonSql', 'tool_repair', $repairId, '" . db_escape($date) . "', $userId)
            ");

            db_query("UPDATE tool_repairs
                      SET status='written_off', finished_date='" . db_escape($date) . "'
                      WHERE id=$repairId");

            log_action($userId, 'update', 'tool_repair', $repairId, [
                'status' => 'written_off',
                'finished_date' => $date,
                'reason' => $comment
            ]);

            header('Location: tool_repairs.php?saved=1');
            exit;
        }
    }
}

if (isset($_GET['saved'])) $success = 'Сохранено';

$repairs = db_fetch_all("
    SELECT tr.*, i.name AS item_name
    FROM tool_repairs tr
    JOIN items i ON i.id = tr.item_id
    ORDER BY tr.status, tr.started_date DESC, tr.id DESC
");

include '../../layout/header.php';
include '../../layout/sidebar.php';
?>

<main class="main-content">

    <div class="card" style="position: sticky; top: 0; z-index: 10; margin-bottom: 16px;">
        <div class="card-body" style="padding: 8px 12px; display:flex; gap:8px; flex-wrap:wrap;">
            <a href="index.php?tab=tool" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">← Назад</a>
        </div>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success"><?php echo h($success); ?></div>
    <?php endif; ?>

    <?php if ($errors): ?>
        <div class="alert alert-danger"><?php echo implode('<br>', array_map('h', $errors)); ?></div>
    <?php endif; ?>

    <div class="card" style="margin-bottom:12px;">
        <div class="card-body" style="max-width:900px;">

            <form method="post">
                <input type="hidden" name="mode" value="start">

                <div style="display:flex; gap:12px; flex-wrap:wrap; align-items:flex-end;">
                    <div style="flex:1; min-width:260px;">
                        <label>Инструмент</label>
                        <select name="item_id" class="form-control">
                            <option value="0">— выбери —</option>
                            <?php foreach ($tools as $t): ?>
                                <option value="<?php echo (int)$t['id']; ?>"><?php echo h($t['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div style="min-width:160px;">
                        <label>Количество</label>
                        <input type="text" name="quantity" class="form-control" placeholder="например 1">
                    </div>

                    <div style="min-width:200px;">
                        <label>Дата начала</label>
                        <input type="date" name="started_date" class="form-control" value="<?php echo h(date('Y-m-d')); ?>">
                    </div>

                    <div style="min-width:220px;">
                        <label>Ориентировочная дата</label>
                        <input type="date" name="expected_date" class="form-control">
                    </div>
                </div>

                <div class="form-group" style="margin:12px 0;">
                    <label>Откуда отправили в ремонт</label>
                    <div style="display:flex; gap:12px; flex-wrap:wrap; margin-top:6px;">
                        <label><input type="radio" name="source_type" value="location" checked> Локация</label>
                        <label><input type="radio" name="source_type" value="workshop"> Цех</label>
                        <label><input type="radio" name="source_type" value="employee"> Сотрудник</label>
                    </div>
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Локация</label>
                    <select name="from_location_id" class="form-control">
                        <option value="0">— выбери —</option>
                        <?php foreach ($visibleLocations as $loc): ?>
                            <option value="<?php echo (int)$loc['id']; ?>"><?php echo h($loc['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Цех</label>
                    <select name="from_workshop_id" class="form-control">
                        <option value="0">— выбери —</option>
                        <?php foreach ($visibleWorkshops as $w): ?>
                            <option value="<?php echo (int)$w['id']; ?>"><?php echo h(($w['location_name'] ?? '') . ($w['location_name'] ? ' — ' : '') . $w['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Сотрудник</label>
                    <select name="from_employee_id" class="form-control">
                        <option value="0">— выбери —</option>
                        <?php foreach ($employees as $e): ?>
                            <option value="<?php echo (int)$e['id']; ?>"><?php echo h($e['full_name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Причина/комментарий (обязательно)</label>
                    <textarea name="reason" class="form-control" rows="3" placeholder="например: сломался, нужна замена детали"></textarea>
                </div>

                <button class="btn btn-success" style="padding:6px 12px;font-size:13px;">Отправить в ремонт</button>
            </form>

        </div>
    </div>

    <div class="card">
        <div class="card-body">

            <table class="table table-striped table-hover">
                <thead>
                    <tr>
                        <th style="width:70px;">ID</th>
                        <th>Инструмент</th>
                        <th style="width:120px;">Кол-во</th>
                        <th style="width:140px;">Статус</th>
                        <th style="width:140px;">Начато</th>
                        <th style="width:160px;">Ожидаем</th>
                        <th style="width:220px;">Действия</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!$repairs): ?>
                        <tr><td colspan="7" style="text-align:center;color:var(--text-secondary);padding:20px;">Ремонтов нет</td></tr>
                    <?php else: ?>
                        <?php foreach ($repairs as $r): ?>
                            <tr>
                                <td><?php echo (int)$r['id']; ?></td>
                                <td><?php echo h($r['item_name']); ?></td>
                                <td><?php echo h(fmt_qty_local($r['quantity'])); ?></td>
                                <td><?php echo h($r['status']); ?></td>
                                <td><?php echo h($r['started_date']); ?></td>
                                <td><?php echo h($r['expected_date'] ?: '—'); ?></td>
                                <td>
                                    <?php if ($r['status'] === 'in_repair'): ?>
                                        <form method="post" style="display:inline-block; margin-right:6px;">
                                            <input type="hidden" name="mode" value="finish_return">
                                            <input type="hidden" name="repair_id" value="<?php echo (int)$r['id']; ?>">
                                            <input type="hidden" name="finish_date" value="<?php echo h(date('Y-m-d')); ?>">
                                            <input type="hidden" name="comment" value="">
                                            <button class="btn btn-secondary" style="padding:4px 8px;font-size:12px;"
                                                    onclick="return confirm('Вернуть из ремонта в то же место, откуда сдали?');">
                                                Вернуть
                                            </button>
                                        </form>

                                        <form method="post" style="display:inline-block;">
                                            <input type="hidden" name="mode" value="finish_writeoff">
                                            <input type="hidden" name="repair_id" value="<?php echo (int)$r['id']; ?>">
                                            <input type="hidden" name="finish_date" value="<?php echo h(date('Y-m-d')); ?>">
                                            <input type="hidden" name="comment" value="Списано при ремонте">
                                            <button class="btn btn-danger" style="padding:4px 8px;font-size:12px;"
                                                    onclick="return confirm('Списать в ремонте?');">
                                                Списать
                                            </button>
                                        </form>
                                    <?php else: ?>
                                        —
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <div style="margin-top:10px;color:var(--text-secondary);font-size:13px;">
                “Вернуть” возвращает инструмент туда же (локация/цех/сотрудник), откуда он был сдан в ремонт.
            </div>

        </div>
    </div>

</main>

<?php include '../../layout/footer.php'; ?>
