<?php
/**
 * Экспорт остатков склада в XLSX без сторонних библиотек.
 * Делает отдельный лист на каждую локацию (удобно печатать).
 *
 * Параметр 'consumable' должен быть: material / tool / consumable
 */

require_once '../../config.php';
require_once '../../auth/guard.php';
require_once '../../lib/db.php';
require_once '../../lib/acl.php';

requirePermission('warehouse', 'view');

require_once __DIR__ . '/../../lib/simple_xlsx_writer.php';

function fmt_qty_local($v) {
    $s = (string)$v;
    if (strpos($s, '.') !== false) $s = rtrim(rtrim($s, '0'), '.');
    return $s === '' ? '0' : $s;
}

$itemType = 'consumable';

// доступные локации (если не закреплена — показываем все активные)
$userId = (int)($_SESSION['user_id'] ?? 0);
$visibleLocations = ACL::getUserLocations($userId);
if (!$visibleLocations) {
    $visibleLocations = db_fetch_all("SELECT id, name FROM locations WHERE is_active=1 ORDER BY name");
}
$locIds = array_map('intval', array_column($visibleLocations, 'id'));
$locIdsStr = $locIds ? implode(',', $locIds) : '0';

// товары для вкладки
$items = db_fetch_all("
    SELECT i.id, i.name, i.unit, mc.name AS category_name
    FROM items i
    LEFT JOIN material_categories mc ON mc.id = i.category_id
    WHERE i.is_active=1 AND i.item_type='" . db_escape($itemType) . "'
    ORDER BY mc.name, i.name
");

// подсчёт остатков по локациям: 1-в-1 как в modules/warehouse/index.php
$balances = []; // [item_id][location_id] = qty
if ($items && $locIds) {
    $itemIds = array_map('intval', array_column($items, 'id'));
    $itemIdsStr = implode(',', $itemIds);

    // Вход в локацию: receipt, transfer, assembly
    $inRows = db_fetch_all("
        SELECT item_id, to_location_id AS location_id, SUM(quantity) AS qty
        FROM inventory_movements
        WHERE to_location_id IN ($locIdsStr)
          AND item_id IN ($itemIdsStr)
          AND item_type='" . db_escape($itemType) . "'
          AND movement_type IN ('receipt','transfer','assembly')
        GROUP BY item_id, to_location_id
    ");

    foreach ($inRows as $r) {
        $iid = (int)$r['item_id'];
        $lid = (int)$r['location_id'];
        $balances[$iid][$lid] = ($balances[$iid][$lid] ?? 0) + (float)$r['qty'];
    }

    // Выход из локации: transfer, deduction, writeoff, assembly
    $outRows = db_fetch_all("
        SELECT item_id, from_location_id AS location_id, SUM(quantity) AS qty
        FROM inventory_movements
        WHERE from_location_id IN ($locIdsStr)
          AND item_id IN ($itemIdsStr)
          AND item_type='" . db_escape($itemType) . "'
          AND movement_type IN ('transfer','deduction','writeoff','assembly')
        GROUP BY item_id, from_location_id
    ");

    foreach ($outRows as $r) {
        $iid = (int)$r['item_id'];
        $lid = (int)$r['location_id'];
        $balances[$iid][$lid] = ($balances[$iid][$lid] ?? 0) - (float)$r['qty'];
    }
}

// Формируем XLSX: 1 лист = 1 локация
$xlsx = new SimpleXlsxWriter();

foreach ($visibleLocations as $loc) {
    $lid = (int)$loc['id'];

    $rows = [];
    $rows[] = ['ID', 'Наименование', 'Категория', 'Ед.', 'Остаток'];

    foreach ($items as $it) {
        $iid = (int)$it['id'];
        $q = (float)($balances[$iid][$lid] ?? 0);
        if ($q == 0.0) continue;

        $rows[] = [
            $iid,
            $it['name'],
            ($it['category_name'] ?: '—'),
            ($it['unit'] ?: 'шт'),
            fmt_qty_local($q),
        ];
    }

    // Если по локации вообще пусто — всё равно лист оставим (чтобы было видно, что выгрузка сделана)
    if (count($rows) === 1) {
        $rows[] = ['—', 'Нет остатков', '—', '—', '0'];
    }

    $xlsx->addSheet($loc['name'], $rows);
}

$human = ($itemType === 'material') ? 'materials' : (($itemType === 'tool') ? 'tools' : 'consumables');
$xlsx->output('stock_' . $human . '_' . date('Ymd_His') . '.xlsx');
