<?php
/**
 * Настройки → Операции → Создание / редактирование
 * Файл: modules/settings/operation_edit.php
 */

require_once '../../config.php';
require_once '../../auth/guard.php';
require_once '../../lib/db.php';
require_once '../../lib/acl.php';

requirePermission('settings', 'view');

$operationId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];

// локальный форматтер количества (без .000)
function fmt_qty_local($v) {
    $s = (string)$v;
    if (strpos($s, '.') !== false) {
        $s = rtrim(rtrim($s, '0'), '.');
    }
    return $s === '' ? '0' : $s;
}

// справочники
$workshops = db_fetch_all("
    SELECT w.id, w.name, l.name AS location_name
    FROM workshops w
    JOIN locations l ON l.id = w.location_id
    WHERE w.is_active = 1 AND l.is_active = 1
    ORDER BY l.name, w.name
");

$items = db_fetch_all("
    SELECT id, name, unit, item_type
    FROM items
    WHERE is_active = 1
    ORDER BY name
");

// данные операции
$operation = [
    'name' => '',
    'workshop_id' => 0,
    'daily_plan_per_employee' => 0,
    'price' => '0.00',
];

// материалы операции (массив строк)
$opMaterials = []; // каждый элемент: ['item_id'=>.., 'quantity'=>..]

// загрузка операции
if ($operationId > 0) {
    $row = db_fetch_one("SELECT * FROM operations WHERE id = $operationId");
    if (!$row) {
        die('Операция не найдена');
    }
    $operation = $row;

    $rows = db_fetch_all("
        SELECT om.item_id, om.quantity, i.name, i.unit
        FROM operation_materials om
        JOIN items i ON i.id = om.item_id
        WHERE om.operation_id = $operationId
        ORDER BY i.name
    ");
    foreach ($rows as $r) {
        $opMaterials[] = [
            'item_id' => (int)$r['item_id'],
            'quantity' => fmt_qty_local($r['quantity']),
        ];
    }
}

// сохранение
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requirePermission('settings', 'edit');

    $name = trim($_POST['name'] ?? '');
    $workshopId = (int)($_POST['workshop_id'] ?? 0);
    $dailyPlan = (int)($_POST['daily_plan_per_employee'] ?? 0);
    $price = trim($_POST['price'] ?? '0');

    // материалы из формы
    $itemIds = isset($_POST['item_id']) ? (array)$_POST['item_id'] : [];
    $qtys    = isset($_POST['quantity']) ? (array)$_POST['quantity'] : [];

    if ($name === '') $errors[] = 'Название операции обязательно';
    if ($workshopId <= 0) $errors[] = 'Цех обязателен';
    if ($dailyPlan < 0) $errors[] = 'Дневной план не может быть отрицательным';
    if ($price === '') $price = '0';

    // простая проверка цены
    $priceNorm = str_replace(',', '.', $price);
    if (!is_numeric($priceNorm)) {
        $errors[] = 'Стоимость должна быть числом';
    }

    // подготовим материалы (отфильтруем пустые строки)
    $newMaterials = [];
    $count = max(count($itemIds), count($qtys));
    for ($i = 0; $i < $count; $i++) {
        $iid = isset($itemIds[$i]) ? (int)$itemIds[$i] : 0;
        $q   = isset($qtys[$i]) ? trim((string)$qtys[$i]) : '';

        if ($iid <= 0) continue;
        if ($q === '') continue;

        $qNorm = str_replace(',', '.', $q);
        if (!is_numeric($qNorm)) {
            $errors[] = 'Количество материала должно быть числом';
            break;
        }

        $qVal = (float)$qNorm;
        if ($qVal <= 0) {
            $errors[] = 'Количество материала должно быть больше нуля';
            break;
        }

        // если одинаковый item повторили несколько раз — суммируем
        if (!isset($newMaterials[$iid])) {
            $newMaterials[$iid] = 0.0;
        }
        $newMaterials[$iid] += $qVal;
    }

    if (!$errors) {
        $nameSql = db_escape($name);
        $priceSql = db_escape(number_format((float)$priceNorm, 2, '.', ''));

        if ($operationId > 0) {
            db_query("
                UPDATE operations
                SET name = '$nameSql',
                    workshop_id = $workshopId,
                    daily_plan_per_employee = $dailyPlan,
                    price = '$priceSql'
                WHERE id = $operationId
            ");
        } else {
            db_query("
                INSERT INTO operations (name, workshop_id, daily_plan_per_employee, price)
                VALUES ('$nameSql', $workshopId, $dailyPlan, '$priceSql')
            ");
            $last = db_fetch_one("SELECT LAST_INSERT_ID() AS id");
            $operationId = $last ? (int)$last['id'] : 0;
        }

        // перезаписываем материалы операции
        db_query("DELETE FROM operation_materials WHERE operation_id = $operationId");

        foreach ($newMaterials as $iid => $qVal) {
            $iid = (int)$iid;
            $qSql = db_escape(number_format($qVal, 3, '.', '')); // храним с точностью, в UI покажем без .000
            db_query("
                INSERT INTO operation_materials (operation_id, item_id, quantity)
                VALUES ($operationId, $iid, '$qSql')
            ");
        }

        header('Location: operations.php');
        exit;
    }

    // если ошибки — заполняем назад
    $operation['name'] = $name;
    $operation['workshop_id'] = $workshopId;
    $operation['daily_plan_per_employee'] = $dailyPlan;
    $operation['price'] = $priceNorm;

    $opMaterials = [];
    foreach ($newMaterials as $iid => $qVal) {
        $opMaterials[] = [
            'item_id' => (int)$iid,
            'quantity' => fmt_qty_local(number_format($qVal, 3, '.', '')),
        ];
    }
}

include '../../layout/header.php';
include '../../layout/sidebar.php';
?>

<main class="main-content">

    <!-- Sticky панель -->
    <div class="card" style="position: sticky; top: 0; z-index: 10; margin-bottom: 16px;">
        <div class="card-body" style="padding: 8px 12px;">
            <a href="operations.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">
                ← Назад к операциям
            </a>
        </div>
    </div>

    <div class="card">
        <div class="card-body">

            <?php if ($errors): ?>
                <div class="alert alert-danger">
                    <?php echo implode('<br>', $errors); ?>
                </div>
            <?php endif; ?>

            <form method="post">

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Название операции</label>
                    <input type="text" name="name" class="form-control"
                           value="<?php echo htmlspecialchars($operation['name'] ?? ''); ?>">
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Цех</label>
                    <select name="workshop_id" class="form-control">
                        <option value="0">— выбери цех —</option>
                        <?php foreach ($workshops as $w): ?>
                            <option value="<?php echo (int)$w['id']; ?>"
                                <?php echo ((int)($operation['workshop_id'] ?? 0) === (int)$w['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($w['location_name'] . ' — ' . $w['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Дневной план на 1 сотрудника (целое число)</label>
                    <input type="number" name="daily_plan_per_employee" class="form-control"
                           value="<?php echo (int)($operation['daily_plan_per_employee'] ?? 0); ?>">
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Стоимость за 1 операцию, ₽</label>
                    <input type="text" name="price" class="form-control"
                           value="<?php echo htmlspecialchars($operation['price'] ?? '0.00'); ?>">
                </div>

                <div class="form-group" style="margin-bottom:12px;">
                    <label>Материалы / инструменты / расходники для операции</label>

                    <div style="margin-top:8px; border: 1px solid rgba(0,0,0,0.1); padding: 8px; border-radius: 6px;">
                        <?php
                        // покажем минимум 5 строк для удобства
                        $rowsToShow = max(5, count($opMaterials));
                        for ($i = 0; $i < $rowsToShow; $i++):
                            $rowItemId = $opMaterials[$i]['item_id'] ?? 0;
                            $rowQty = $opMaterials[$i]['quantity'] ?? '';
                        ?>
                            <div style="display:flex; gap:8px; margin-bottom:8px; align-items:center;">
                                <select name="item_id[]" class="form-control" style="flex: 1;">
                                    <option value="0">— выбери позицию —</option>
                                    <?php foreach ($items as $it): ?>
                                        <?php
                                            $iid = (int)$it['id'];
                                            $txtType = $it['item_type'] === 'material' ? 'Материал' : ($it['item_type'] === 'tool' ? 'Инструмент' : 'Расходник');
                                            $title = $it['name'] . ' (' . $txtType . ', ' . $it['unit'] . ')';
                                        ?>
                                        <option value="<?php echo $iid; ?>" <?php echo ($rowItemId === $iid) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($title); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>

                                <input type="text" name="quantity[]" class="form-control" style="width:160px;"
                                       placeholder="Кол-во"
                                       value="<?php echo htmlspecialchars($rowQty); ?>">
                            </div>
                        <?php endfor; ?>

                        <div style="color: var(--text-secondary); font-size: 13px; margin-top: 6px;">
                            Если строка пустая — она не сохранится. Количество показывается без “.000”.
                        </div>
                    </div>
                </div>

                <button class="btn btn-success" style="padding:6px 12px;font-size:13px;">
                    Сохранить
                </button>

            </form>

        </div>
    </div>

</main>

<?php include '../../layout/footer.php'; ?>
