<?php
/**
 * Настройки → Логи
 * Файл: modules/settings/logs.php
 *
 * По умолчанию: последние 31 день
 * Только чтение
 * Выгрузка в Excel: CSV (Excel открывает без проблем)
 */

require_once '../../config.php';
require_once '../../auth/guard.php';
require_once '../../lib/db.php';
require_once '../../lib/acl.php';

requirePermission('settings', 'view');

function h($s) { return htmlspecialchars((string)$s); }

function parse_json_maybe($s) {
    if ($s === null) return null;
    $s = trim((string)$s);
    if ($s === '') return null;
    $j = json_decode($s, true);
    return is_array($j) ? $j : null;
}

/**
 * Делаем человеческое сообщение (не техническое).
 */
function build_message($row, $itemNameMap, $productNameMap) {
    $action = $row['action'] ?? '';
    $module = $row['module'] ?? '';
    $rtype  = $row['record_type'] ?? '';
    $rid    = (int)($row['record_id'] ?? 0);

    $new = parse_json_maybe($row['new_values'] ?? '');
    $old = parse_json_maybe($row['old_values'] ?? '');

    $comment = '';
    if (is_array($new)) {
        if (!empty($new['comment'])) $comment = (string)$new['comment'];
        if ($comment === '' && !empty($new['reason'])) $comment = (string)$new['reason'];
    }
    if ($comment === '' && is_array($old)) {
        if (!empty($old['comment'])) $comment = (string)$old['comment'];
        if ($comment === '' && !empty($old['reason'])) $comment = (string)$old['reason'];
    }

    // Склад (движения)
    if (is_array($new) && ($module === 'warehouse' || ($rtype === 'inventory_movement'))) {
        $mt = $new['movement_type'] ?? '';
        $it = $new['item_type'] ?? '';
        $iid = isset($new['item_id']) ? (int)$new['item_id'] : 0;
        $qty = isset($new['quantity']) ? $new['quantity'] : null;

        $name = '';
        if ($it === 'product') {
            if ($iid && isset($productNameMap[$iid])) $name = $productNameMap[$iid];
        } else {
            if ($iid && isset($itemNameMap[$iid])) $name = $itemNameMap[$iid];
        }

        $mtRu = $mt;
        if ($mt === 'receipt') $mtRu = 'Приход';
        elseif ($mt === 'deduction') $mtRu = 'Выдача/расход';
        elseif ($mt === 'writeoff') $mtRu = 'Списание';
        elseif ($mt === 'transfer') $mtRu = 'Перемещение';
        elseif ($mt === 'assembly') $mtRu = 'Сборка';
        elseif ($mt === 'shipment') $mtRu = 'Отгрузка';

        $parts = [];
        if ($mtRu) $parts[] = $mtRu;
        if ($name !== '') $parts[] = $name;
        if ($qty !== null && $qty !== '') $parts[] = 'кол-во: ' . (string)$qty;
        if ($comment !== '') $parts[] = 'комментарий: ' . $comment;

        if ($parts) return implode(' — ', $parts);
    }

    // Ремонт инструмента
    if (is_array($new) && ($rtype === 'tool_repair' || $module === 'warehouse')) {
        if (($new['status'] ?? '') === 'returned') {
            $msg = 'Возврат инструмента из ремонта';
            if ($comment !== '') $msg .= ' — ' . $comment;
            return $msg;
        }
        if (($new['status'] ?? '') === 'written_off') {
            $msg = 'Списание инструмента в ремонте';
            if ($comment !== '') $msg .= ' — ' . $comment;
            return $msg;
        }
    }

    // Общее
    $aRu = $action;
    if ($action === 'create') $aRu = 'Создано';
    elseif ($action === 'update') $aRu = 'Изменено';
    elseif ($action === 'delete') $aRu = 'Удалено';
    elseif ($action === 'read') $aRu = 'Просмотр';

    $mRu = $module;
    if ($module === 'auth') $mRu = 'Авторизация';
    elseif ($module === 'warehouse') $mRu = 'Склад';

    $msg = $aRu;
    if ($rtype) $msg .= ' ' . $rtype;
    if ($rid) $msg .= ' #' . $rid;
    if ($comment !== '') $msg .= ' — ' . $comment;
    if ($msg === '') $msg = 'Событие';
    if ($mRu) $msg = $mRu . ': ' . $msg;

    return $msg;
}

// фильтры
$days = 31;
$from = $_GET['from'] ?? '';
$to = $_GET['to'] ?? '';

if ($to === '') $to = date('Y-m-d');
if ($from === '') $from = date('Y-m-d', time() - ($days * 86400));

$moduleFilter = $_GET['module'] ?? '';
$userFilter = (int)($_GET['user_id'] ?? 0);

$where = [];
$where[] = "ul.created_at >= '" . db_escape($from . " 00:00:00") . "'";
$where[] = "ul.created_at <= '" . db_escape($to . " 23:59:59") . "'";

if ($moduleFilter !== '') $where[] = "ul.module='" . db_escape($moduleFilter) . "'";
if ($userFilter > 0) $where[] = "ul.user_id=" . (int)$userFilter;

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

// справочники
$itemNameMap = [];
$productNameMap = [];

$items = db_fetch_all("SELECT id, name FROM items");
foreach ($items as $r) $itemNameMap[(int)$r['id']] = $r['name'];

$products = db_fetch_all("SELECT id, name FROM products");
foreach ($products as $r) $productNameMap[(int)$r['id']] = $r['name'];

// списки для фильтров
$modules = db_fetch_all("SELECT DISTINCT module FROM user_logs ORDER BY module");
$users = db_fetch_all("SELECT id, full_name FROM users WHERE is_active=1 ORDER BY full_name");

// логи
$logs = db_fetch_all("
    SELECT ul.*, u.full_name
    FROM user_logs ul
    LEFT JOIN users u ON u.id = ul.user_id
    $whereSql
    ORDER BY ul.created_at DESC
    LIMIT 2000
");

// экспорт CSV (Excel)
if (isset($_GET['export']) && $_GET['export'] == '1') {
    $filename = "logs_" . date('Ymd_His') . ".csv";
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename=' . $filename);

    // BOM для Excel
    echo "\xEF\xBB\xBF";

    $out = fopen('php://output', 'w');
    fputcsv($out, ['Дата', 'Пользователь (ФИО)', 'Модуль', 'Действие', 'Сообщение'], ';');

    foreach ($logs as $row) {
        $msg = build_message($row, $itemNameMap, $productNameMap);
        $dt = $row['created_at'] ?? '';
        $fn = $row['full_name'] ?: ('ID ' . (int)$row['user_id']);
        fputcsv($out, [$dt, $fn, $row['module'], $row['action'], $msg], ';');
    }
    fclose($out);
    exit;
}

include '../../layout/header.php';
include '../../layout/sidebar.php';
?>

<main class="main-content">

    <!-- Sticky панель настроек -->
    <div class="card" style="position: sticky; top: 0; z-index: 10; margin-bottom: 16px;">
        <div class="card-body" style="padding: 8px 12px;">
            <div style="display:flex; flex-wrap:wrap; gap:8px;">
                <a href="users.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Пользователи</a>
                <a href="roles.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Роли</a>
                <a href="locations.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Локации</a>
                <a href="workshops.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Цеха</a>
                <a href="operations.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Операции</a>
                <a href="material_categories.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Категории материалов</a>
                <a href="tg_bot.php" class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">TG-бот</a>
                <a href="logs.php" class="btn btn-primary" style="padding:6px 12px;font-size:13px;">Логи</a>
            </div>
        </div>
    </div>

    <!-- Фильтры -->
    <div class="card" style="margin-bottom:12px;">
        <div class="card-body" style="padding: 8px 12px;">
            <form method="get" style="display:flex; gap:10px; flex-wrap:wrap; align-items:flex-end;">
                <div style="min-width:160px;">
                    <label style="font-size:13px;">С</label>
                    <input type="date" name="from" class="form-control" value="<?php echo h($from); ?>">
                </div>
                <div style="min-width:160px;">
                    <label style="font-size:13px;">По</label>
                    <input type="date" name="to" class="form-control" value="<?php echo h($to); ?>">
                </div>
                <div style="min-width:180px;">
                    <label style="font-size:13px;">Модуль</label>
                    <select name="module" class="form-control">
                        <option value="">Все</option>
                        <?php foreach ($modules as $m): ?>
                            <?php $val = (string)$m['module']; ?>
                            <option value="<?php echo h($val); ?>" <?php echo ($moduleFilter===$val?'selected':''); ?>>
                                <?php echo h($val); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div style="min-width:220px;">
                    <label style="font-size:13px;">Пользователь</label>
                    <select name="user_id" class="form-control">
                        <option value="0">Все</option>
                        <?php foreach ($users as $u): ?>
                            <option value="<?php echo (int)$u['id']; ?>" <?php echo ($userFilter===(int)$u['id']?'selected':''); ?>>
                                <?php echo h($u['full_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <button class="btn btn-secondary" style="padding:6px 12px;font-size:13px;">Показать</button>

                <a class="btn btn-secondary" style="padding:6px 12px;font-size:13px;"
                   href="?from=<?php echo h($from); ?>&to=<?php echo h($to); ?>&module=<?php echo h($moduleFilter); ?>&user_id=<?php echo (int)$userFilter; ?>&export=1">
                    Выгрузить в Excel
                </a>
            </form>
        </div>
    </div>

    <!-- Таблица -->
    <div class="card">
        <div class="card-body">

            <table class="table table-striped table-hover">
                <thead>
                    <tr>
                        <th style="width:170px;">Дата</th>
                        <th style="width:240px;">Пользователь (ФИО)</th>
                        <th style="width:120px;">Модуль</th>
                        <th style="width:110px;">Действие</th>
                        <th>Сообщение</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!$logs): ?>
                        <tr>
                            <td colspan="5" style="text-align:center;color:var(--text-secondary);padding:20px;">
                                Логи за выбранный период не найдены
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($logs as $row): ?>
                            <?php
                                $msg = build_message($row, $itemNameMap, $productNameMap);
                                $fio = $row['full_name'] ?: ('ID ' . (int)$row['user_id']);
                            ?>
                            <tr>
                                <td><?php echo h($row['created_at']); ?></td>
                                <td><?php echo h($fio); ?></td>
                                <td><?php echo h($row['module']); ?></td>
                                <td><?php echo h($row['action']); ?></td>
                                <td><?php echo h($msg); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <div style="margin-top:10px;color:var(--text-secondary);font-size:13px;">
                По умолчанию показываются логи за последние 31 день. Раздел только для просмотра.
            </div>

        </div>
    </div>

</main>

<?php include '../../layout/footer.php'; ?>
