<?php
/**
 * Минимальный генератор XLSX без внешних библиотек.
 * Возможности:
 * - 1+ листов
 * - жирная шапка (первая строка)
 * - заморозка первой строки
 * - настройки печати: альбомная, уместить по ширине страницы
 * - авто-подбор ширины колонок (приблизительно по длине текста)
 * - границы (контур) для заполненных ячеек
 *
 * Требование на сервере: включено расширение PHP ZipArchive (zip).
 */

class SimpleXlsxWriter
{
    private array $sheets = [];          // [['name'=>..., 'rows'=>...], ...]
    private array $sharedStrings = [];   // value => index
    private array $sharedList = [];      // index => value

    public function addSheet(string $name, array $rows): void
    {
        $name = trim($name);
        if ($name === '') $name = 'Лист';

        // Excel: имя листа <= 31 символ, запрещены: []*/\:? 
        $name = preg_replace('/[\\[\\]\\*\\?\\/\\\\:]/u', ' ', $name);
        $name = mb_substr($name, 0, 31);

        $this->sheets[] = ['name' => $name, 'rows' => $rows];
    }

    public function output(string $filename): void
    {
        if (!class_exists('ZipArchive')) {
            $this->fail('На сервере нет расширения PHP ZipArchive. Нужно включить расширение zip.');
        }
        if (!$this->sheets) {
            $this->fail('Нет данных для экспорта.');
        }

        $this->buildSharedStrings();

        $tmp = tempnam(sys_get_temp_dir(), 'xlsx_');
        $zip = new ZipArchive();
        if ($zip->open($tmp, ZipArchive::OVERWRITE) !== true) {
            $this->fail('Не удалось создать XLSX-архив.');
        }

        // docProps
        $zip->addFromString('docProps/app.xml', $this->xmlAppProps());
        $zip->addFromString('docProps/core.xml', $this->xmlCoreProps());

        // relationships root
        $zip->addFromString('_rels/.rels', $this->xmlRelsRoot());

        // workbook + rels
        $zip->addFromString('xl/workbook.xml', $this->xmlWorkbook());
        $zip->addFromString('xl/_rels/workbook.xml.rels', $this->xmlWorkbookRels());

        // styles + sharedStrings
        $zip->addFromString('xl/styles.xml', $this->xmlStyles());
        $zip->addFromString('xl/sharedStrings.xml', $this->xmlSharedStrings());

        // sheets
        for ($i = 0; $i < count($this->sheets); $i++) {
            $zip->addFromString('xl/worksheets/sheet' . ($i + 1) . '.xml', $this->xmlSheet($i));
        }

        // content types
        $zip->addFromString('[Content_Types].xml', $this->xmlContentTypes());

        $zip->close();

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename=' . $this->safeFilename($filename));
        header('Cache-Control: max-age=0');
        header('Pragma: public');
        header('Expires: 0');

        readfile($tmp);
        @unlink($tmp);
        exit;
    }

    /* -------------------- helpers -------------------- */

    private function fail(string $msg): void
    {
        http_response_code(500);
        echo $msg;
        exit;
    }

    private function safeFilename(string $name): string
    {
        $name = preg_replace('/[^A-Za-z0-9._-]+/u', '_', $name);
        if ($name === '' || $name === '_') $name = 'export.xlsx';
        if (!str_ends_with(strtolower($name), '.xlsx')) $name .= '.xlsx';
        return $name;
    }

    private function buildSharedStrings(): void
    {
        $this->sharedStrings = [];
        $this->sharedList = [];

        foreach ($this->sheets as $sheet) {
            foreach ($sheet['rows'] as $row) {
                foreach ($row as $cell) {
                    if ($cell === null) continue;
                    if ($this->isNumeric($cell)) continue;
                    $v = (string)$cell;
                    $this->getSharedStringIndex($v);
                }
            }
        }
    }

    private function isNumeric($v): bool
    {
        if (is_int($v) || is_float($v)) return true;
        if (!is_string($v)) return false;
        $s = trim($v);
        if ($s === '') return false;
        if (!preg_match('/^-?\d+(\.\d+)?$/', $s)) return false;
        // не считаем числом строки с ведущими нулями (например "001")
        if (preg_match('/^-?0\d+/', $s)) return false;
        return true;
    }

    private function getSharedStringIndex(string $v): int
    {
        if (isset($this->sharedStrings[$v])) return $this->sharedStrings[$v];
        $idx = count($this->sharedList);
        $this->sharedStrings[$v] = $idx;
        $this->sharedList[] = $v;
        return $idx;
    }

    private function xmlEscape(string $s): string
    {
        return htmlspecialchars($s, ENT_QUOTES | ENT_XML1, 'UTF-8');
    }

    private function colName(int $n): string
    {
        $s = '';
        while ($n > 0) {
            $n--;
            $s = chr(($n % 26) + 65) . $s;
            $n = intdiv($n, 26);
        }
        return $s;
    }

    private function dimensionRef(array $rows): string
    {
        $maxCols = 0;
        $maxRows = count($rows);
        foreach ($rows as $r) $maxCols = max($maxCols, is_array($r) ? count($r) : 0);
        if ($maxCols < 1) $maxCols = 1;
        if ($maxRows < 1) $maxRows = 1;
        $endCol = $this->colName($maxCols);
        return 'A1:' . $endCol . $maxRows;
    }

    private function calcColWidths(array $rows, int $maxCols): array
    {
        // Примерный расчёт ширины в "Excel units" (символы Calibri 11)
        $maxLen = array_fill(0, $maxCols, 0);

        foreach ($rows as $r) {
            if (!is_array($r)) $r = [];
            for ($c = 0; $c < $maxCols; $c++) {
                $val = $r[$c] ?? '';
                if ($val === null) $val = '';
                $s = (string)$val;

                // Для чисел берём строку как есть (после fmt_qty у вас уже нет .000)
                // Убираем переводы строк (на всякий)
                $s = str_replace(["\r", "\n"], ' ', $s);
                $len = mb_strlen($s);

                if ($len > $maxLen[$c]) $maxLen[$c] = $len;
            }
        }

        $widths = [];
        for ($c = 0; $c < $maxCols; $c++) {
            // Паддинг + ограничение (чтобы не было огромных колонок)
            $w = $maxLen[$c] + 2;
            if ($w < 8) $w = 8;
            if ($w > 60) $w = 60;
            $widths[$c] = $w;
        }
        return $widths;
    }

    /* -------------------- XML -------------------- */

    private function xmlContentTypes(): string
    {
        $overrides = '';
        $sheetCount = count($this->sheets);
        for ($i = 1; $i <= $sheetCount; $i++) {
            $overrides .= '<Override PartName="/xl/worksheets/sheet' . $i . '.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>';
        }

        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">'
            . '<Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>'
            . '<Default Extension="xml" ContentType="application/xml"/>'
            . '<Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>'
            . $overrides
            . '<Override PartName="/xl/styles.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml"/>'
            . '<Override PartName="/xl/sharedStrings.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml"/>'
            . '<Override PartName="/docProps/core.xml" ContentType="application/vnd.openxmlformats-package.core-properties+xml"/>'
            . '<Override PartName="/docProps/app.xml" ContentType="application/vnd.openxmlformats-officedocument.extended-properties+xml"/>'
            . '</Types>';
    }

    private function xmlRelsRoot(): string
    {
        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">'
            . '<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>'
            . '<Relationship Id="rId2" Type="http://schemas.openxmlformats.org/package/2006/relationships/metadata/core-properties" Target="docProps/core.xml"/>'
            . '<Relationship Id="rId3" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/extended-properties" Target="docProps/app.xml"/>'
            . '</Relationships>';
    }

    private function xmlAppProps(): string
    {
        $titles = implode('', array_map(fn($s) => '<vt:lpstr>' . $this->xmlEscape($s['name']) . '</vt:lpstr>', $this->sheets));
        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<Properties xmlns="http://schemas.openxmlformats.org/officeDocument/2006/extended-properties" xmlns:vt="http://schemas.openxmlformats.org/officeDocument/2006/docPropsVTypes">'
            . '<Application>ASCRM</Application>'
            . '<DocSecurity>0</DocSecurity>'
            . '<ScaleCrop>false</ScaleCrop>'
            . '<HeadingPairs><vt:vector size="2" baseType="variant"><vt:variant><vt:lpstr>Worksheets</vt:lpstr></vt:variant><vt:variant><vt:i4>' . count($this->sheets) . '</vt:i4></vt:variant></vt:vector></HeadingPairs>'
            . '<TitlesOfParts><vt:vector size="' . count($this->sheets) . '" baseType="lpstr">'
            . $titles
            . '</vt:vector></TitlesOfParts>'
            . '</Properties>';
    }

    private function xmlCoreProps(): string
    {
        $dt = gmdate('Y-m-d\TH:i:s\Z');
        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<cp:coreProperties xmlns:cp="http://schemas.openxmlformats.org/package/2006/metadata/core-properties" xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:dcterms="http://purl.org/dc/terms/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">'
            . '<dc:title>Экспорт склада</dc:title>'
            . '<dc:creator>ASCRM</dc:creator>'
            . '<cp:lastModifiedBy>ASCRM</cp:lastModifiedBy>'
            . '<dcterms:created xsi:type="dcterms:W3CDTF">' . $dt . '</dcterms:created>'
            . '<dcterms:modified xsi:type="dcterms:W3CDTF">' . $dt . '</dcterms:modified>'
            . '</cp:coreProperties>';
    }

    private function xmlWorkbook(): string
    {
        $sheetsXml = '';
        for ($i = 0; $i < count($this->sheets); $i++) {
            $sid = $i + 1;
            $name = $this->xmlEscape($this->sheets[$i]['name']);
            $sheetsXml .= '<sheet name="' . $name . '" sheetId="' . $sid . '" r:id="rId' . $sid . '"/>';
        }

        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">'
            . '<bookViews><workbookView activeTab="0"/></bookViews>'
            . '<sheets>' . $sheetsXml . '</sheets>'
            . '</workbook>';
    }

    private function xmlWorkbookRels(): string
    {
        $rels = '';
        for ($i = 0; $i < count($this->sheets); $i++) {
            $sid = $i + 1;
            $rels .= '<Relationship Id="rId' . $sid . '" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/sheet' . $sid . '.xml"/>';
        }

        $next = count($this->sheets) + 1;
        $rels .= '<Relationship Id="rId' . $next . '" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/styles" Target="styles.xml"/>';
        $rels .= '<Relationship Id="rId' . ($next + 1) . '" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/sharedStrings" Target="sharedStrings.xml"/>';

        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">'
            . $rels
            . '</Relationships>';
    }

    private function xmlStyles(): string
    {
        // fills минимум 2: none и gray125
        // borders: 0 = без границ, 1 = тонкая рамка со всех сторон
        // cellXfs:
        //   0 обычный
        //   1 шапка: жирный + рамка
        //   2 данные: рамка
        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<styleSheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">'
            . '<fonts count="2">'
            . '<font><sz val="11"/><color theme="1"/><name val="Calibri"/><family val="2"/></font>'
            . '<font><b/><sz val="11"/><color theme="1"/><name val="Calibri"/><family val="2"/></font>'
            . '</fonts>'
            . '<fills count="2">'
            . '<fill><patternFill patternType="none"/></fill>'
            . '<fill><patternFill patternType="gray125"/></fill>'
            . '</fills>'
            . '<borders count="2">'
            . '<border><left/><right/><top/><bottom/><diagonal/></border>'
            . '<border>'
            . '<left style="thin"/><right style="thin"/><top style="thin"/><bottom style="thin"/>'
            . '<diagonal/>'
            . '</border>'
            . '</borders>'
            . '<cellStyleXfs count="1"><xf numFmtId="0" fontId="0" fillId="0" borderId="0"/></cellStyleXfs>'
            . '<cellXfs count="3">'
            . '<xf numFmtId="0" fontId="0" fillId="0" borderId="0" xfId="0" applyFont="1"/>'
            . '<xf numFmtId="0" fontId="1" fillId="0" borderId="1" xfId="0" applyFont="1" applyBorder="1"/>'
            . '<xf numFmtId="0" fontId="0" fillId="0" borderId="1" xfId="0" applyBorder="1"/>'
            . '</cellXfs>'
            . '</styleSheet>';
    }

    private function xmlSharedStrings(): string
    {
        $count = count($this->sharedList);
        $items = '';
        foreach ($this->sharedList as $s) {
            $esc = $this->xmlEscape($s);
            $space = (preg_match('/^\s|\s$/u', $s) ? ' xml:space="preserve"' : '');
            $items .= '<si><t' . $space . '>' . $esc . '</t></si>';
        }

        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<sst xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" count="' . $count . '" uniqueCount="' . $count . '">'
            . $items
            . '</sst>';
    }

    private function xmlSheet(int $i): string
    {
        $rows = $this->sheets[$i]['rows'];

        $maxCols = 0;
        foreach ($rows as $r) $maxCols = max($maxCols, is_array($r) ? count($r) : 0);
        if ($maxCols < 1) $maxCols = 1;

        // авто-ширина колонок (приблизительно)
        $widths = $this->calcColWidths($rows, $maxCols);
        $colsXml = '<cols>';
        for ($c = 0; $c < $maxCols; $c++) {
            $min = $c + 1;
            $max = $c + 1;
            $w = number_format($widths[$c], 2, '.', '');
            $colsXml .= '<col min="' . $min . '" max="' . $max . '" width="' . $w . '" customWidth="1"/>';
        }
        $colsXml .= '</cols>';

        $sheetData = '';
        $rnum = 1;
        foreach ($rows as $r) {
            if (!is_array($r)) $r = [];
            $cellsXml = '';
            for ($c = 0; $c < $maxCols; $c++) {
                $val = $r[$c] ?? '';
                $cellRef = $this->colName($c + 1) . $rnum;

                // стиль: 1-я строка = шапка (1), остальные заполненные ячейки = данные (2)
                $style = '';
                if ($rnum === 1) {
                    $style = ' s="1"';
                } else {
                    if ($val !== null && $val !== '') $style = ' s="2"';
                }

                if ($val === null || $val === '') {
                    $cellsXml .= '<c r="' . $cellRef . '"' . $style . '/>';
                    continue;
                }

                if ($this->isNumeric($val)) {
                    $v = str_replace(',', '.', (string)$val);
                    $cellsXml .= '<c r="' . $cellRef . '"' . $style . '><v>' . $this->xmlEscape($v) . '</v></c>';
                } else {
                    $idx = $this->getSharedStringIndex((string)$val);
                    $cellsXml .= '<c r="' . $cellRef . '" t="s"' . $style . '><v>' . $idx . '</v></c>';
                }
            }
            $sheetData .= '<row r="' . $rnum . '">' . $cellsXml . '</row>';
            $rnum++;
        }

        $sheetViews =
            '<sheetViews><sheetView workbookViewId="0">'
            . '<pane ySplit="1" topLeftCell="A2" activePane="bottomLeft" state="frozen"/>'
            . '</sheetView></sheetViews>';

        $pageMargins = '<pageMargins left="0.7" right="0.7" top="0.75" bottom="0.75" header="0.3" footer="0.3"/>';
        $pageSetup = '<pageSetup orientation="landscape" fitToWidth="1" fitToHeight="0" paperSize="9"/>'; // A4

        $dim = $this->dimensionRef($rows);

        return '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
            . '<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">'
            . '<dimension ref="' . $dim . '"/>'
            . $sheetViews
            . '<sheetFormatPr defaultRowHeight="15"/>'
            . $colsXml
            . '<sheetData>' . $sheetData . '</sheetData>'
            . $pageMargins
            . $pageSetup
            . '</worksheet>';
    }
}
