<?php
/**
 * Библиотека для работы с правами доступа (ACL)
 * Файл: lib/acl.php
 */

class ACL {

    /**
     * Проверка прав доступа пользователя к модулю/действию
     */
    public static function hasPermission($userId, $module, $action = 'view') {
        $userId = (int)$userId;
        $module = db_escape($module);
        $action = db_escape($action);

        $sql = "SELECT COUNT(*) as count
                FROM role_permissions rp
                JOIN permissions p ON rp.permission_id = p.id
                JOIN users u ON u.role_id = rp.role_id
                WHERE u.id = $userId
                AND p.module = '$module'
                AND p.action = '$action'";

        $result = db_fetch_one($sql);
        return $result && (int)$result['count'] > 0;
    }

    /**
     * Получить роль пользователя
     */
    public static function getUserRole($userId) {
        $userId = (int)$userId;

        $sql = "SELECT r.*
                FROM roles r
                JOIN users u ON u.role_id = r.id
                WHERE u.id = $userId";

        return db_fetch_one($sql);
    }

    /**
     * Проверка, является ли пользователь администратором
     */
    public static function isAdmin($userId) {
        $role = self::getUserRole($userId);
        return $role && $role['name'] === 'Администратор';
    }

    /**
     * Проверка, является ли пользователь директором
     */
    public static function isDirector($userId) {
        $role = self::getUserRole($userId);
        return $role && $role['name'] === 'Директор';
    }

    /**
     * Получить локации пользователя
     * Правило: если нет закреплённой локации -> видит все.
     */
    public static function getUserLocations($userId) {
        $userId = (int)$userId;

        // Администратор и директор видят все локации
        if (self::isAdmin($userId) || self::isDirector($userId)) {
            return db_fetch_all("SELECT * FROM locations WHERE is_active = 1");
        }

        $sql = "SELECT l.*
                FROM locations l
                JOIN user_locations ul ON ul.location_id = l.id
                WHERE ul.user_id = $userId AND l.is_active = 1";

        $rows = db_fetch_all($sql);

        // ВАЖНО: если привязки нет -> видит все
        if (empty($rows)) {
            return db_fetch_all("SELECT * FROM locations WHERE is_active = 1");
        }

        return $rows;
    }

    /**
     * Получить ID локаций пользователя
     */
    public static function getUserLocationIds($userId) {
        $locations = self::getUserLocations($userId);
        return array_column($locations, 'id');
    }

    /**
     * Получить цеха пользователя
     * Правило:
     * - если нет закрепленных цехов -> видит все (в рамках доступных локаций)
     * - если нет закрепленных локаций -> видит все локации (см. getUserLocations)
     */
    public static function getUserWorkshops($userId) {
        $userId = (int)$userId;

        // Администратор и директор видят все цеха
        if (self::isAdmin($userId) || self::isDirector($userId)) {
            return db_fetch_all("SELECT * FROM workshops WHERE is_active = 1");
        }

        // Берём доступные локации (если не закреплено -> все)
        $locationIds = self::getUserLocationIds($userId);

        // Проверяем закрепленные цеха
        $sqlAssigned = "SELECT w.*
                        FROM workshops w
                        JOIN user_workshops uw ON uw.workshop_id = w.id
                        WHERE uw.user_id = $userId AND w.is_active = 1";
        $assigned = db_fetch_all($sqlAssigned);

        // Если цеха закреплены -> показываем только их
        if (!empty($assigned)) {
            // на всякий случай фильтруем по доступным локациям
            if (!empty($locationIds)) {
                $allowedLoc = array_map('intval', $locationIds);
                $out = [];
                foreach ($assigned as $w) {
                    if (in_array((int)$w['location_id'], $allowedLoc, true)) {
                        $out[] = $w;
                    }
                }
                return $out;
            }
            return $assigned;
        }

        // Если цеха НЕ закреплены -> показываем все цеха доступных локаций
        if (empty($locationIds)) {
            // по логике сюда почти не попадём, но пусть будет безопасно
            return db_fetch_all("SELECT * FROM workshops WHERE is_active = 1");
        }

        $locationIdsStr = implode(',', array_map('intval', $locationIds));
        return db_fetch_all("SELECT * FROM workshops WHERE location_id IN ($locationIdsStr) AND is_active = 1");
    }

    /**
     * Проверка доступа к локации
     */
    public static function hasAccessToLocation($userId, $locationId) {
        $locationIds = self::getUserLocationIds($userId);
        return in_array((int)$locationId, array_map('intval', $locationIds), true);
    }

    /**
     * Проверка доступа к цеху
     */
    public static function hasAccessToWorkshop($userId, $workshopId) {
        $workshops = self::getUserWorkshops($userId);
        $workshopIds = array_column($workshops, 'id');
        return in_array((int)$workshopId, array_map('intval', $workshopIds), true);
    }

    /**
     * Получить SQL условие для фильтрации по локациям
     * Правило: если нет закреплённой локации -> нет ограничений.
     */
    public static function getLocationFilter($userId, $tableAlias = '') {
        if (self::isAdmin($userId) || self::isDirector($userId)) {
            return '1=1';
        }

        // есть ли реальные закрепления?
        $userId = (int)$userId;
        $cnt = db_fetch_one("SELECT COUNT(*) as c FROM user_locations WHERE user_id = $userId");
        if ($cnt && (int)$cnt['c'] === 0) {
            return '1=1';
        }

        $locationIds = self::getUserLocationIds($userId);
        if (empty($locationIds)) {
            return '1=1';
        }

        $locationIdsStr = implode(',', array_map('intval', $locationIds));
        $prefix = $tableAlias ? $tableAlias . '.' : '';
        return "{$prefix}location_id IN ($locationIdsStr)";
    }
}
?>
