<?php
/**
 * Управление пользователями
 * Файл: modules/settings/users.php
 */

require_once '../../config.php';
require_once '../../auth/guard.php';
require_once '../../lib/db.php';
require_once '../../lib/acl.php';

requirePermission('settings', 'view');

$pageTitle = 'Пользователи - ' . APP_NAME;
$userId = $_SESSION['user_id'];

$message = '';
$error = '';

// Обработка действий
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add' && ACL::hasPermission($userId, 'settings', 'create')) {
        $login = trim($_POST['login'] ?? '');
        $password = $_POST['password'] ?? '';
        $fullName = trim($_POST['full_name'] ?? '');
        $roleId = (int)($_POST['role_id'] ?? 0);
        $locationIds = $_POST['location_ids'] ?? [];
        
        if (empty($login) || empty($password) || empty($fullName) || $roleId === 0) {
            $error = 'Заполните все обязательные поля';
        } else {
            // Проверка уникальности логина
            $checkSql = "SELECT COUNT(*) as count FROM users WHERE login = '" . db_escape($login) . "'";
            if (db_fetch_one($checkSql)['count'] > 0) {
                $error = 'Пользователь с таким логином уже существует';
            } else {
                $passwordHash = password_hash($password, PASSWORD_DEFAULT);
                $loginEsc = db_escape($login);
                $fullNameEsc = db_escape($fullName);
                
                db()->beginTransaction();
                try {
                    $sql = "INSERT INTO users (login, password_hash, full_name, role_id, is_active, created_at) 
                            VALUES ('$loginEsc', '$passwordHash', '$fullNameEsc', $roleId, 1, NOW())";
                    db_query($sql);
                    
                    $newUserId = db()->getLastInsertId();
                    
                    // Привязка к локациям
                    foreach ($locationIds as $locationId) {
                        $locationId = (int)$locationId;
                        db_query("INSERT INTO user_locations (user_id, location_id) VALUES ($newUserId, $locationId)");
                    }
                    
                    db()->commit();
                    $message = 'Пользователь успешно добавлен';
                } catch (Exception $e) {
                    db()->rollback();
                    $error = 'Ошибка при добавлении пользователя';
                }
            }
        }
    }
    
    if ($action === 'toggle' && ACL::hasPermission($userId, 'settings', 'update')) {
        $targetUserId = (int)($_POST['user_id'] ?? 0);
        if ($targetUserId > 0 && $targetUserId !== $userId) {
            $currentStatus = db_fetch_one("SELECT is_active FROM users WHERE id = $targetUserId")['is_active'];
            $newStatus = $currentStatus ? 0 : 1;
            db_query("UPDATE users SET is_active = $newStatus WHERE id = $targetUserId");
            $message = $newStatus ? 'Пользователь активирован' : 'Пользователь деактивирован';
        }
    }
    
    if ($action === 'delete' && ACL::hasPermission($userId, 'settings', 'delete')) {
        $targetUserId = (int)($_POST['user_id'] ?? 0);
        if ($targetUserId > 0 && $targetUserId !== $userId) {
            db_query("DELETE FROM user_locations WHERE user_id = $targetUserId");
            db_query("DELETE FROM user_workshops WHERE user_id = $targetUserId");
            db_query("DELETE FROM users WHERE id = $targetUserId");
            $message = 'Пользователь удален';
        }
    }
}

// Получение списка пользователей
$users = db_fetch_all("
    SELECT u.*, r.name as role_name,
           GROUP_CONCAT(DISTINCT l.name SEPARATOR ', ') as locations
    FROM users u
    JOIN roles r ON r.id = u.role_id
    LEFT JOIN user_locations ul ON ul.user_id = u.id
    LEFT JOIN locations l ON l.id = ul.location_id
    GROUP BY u.id
    ORDER BY u.created_at DESC
");

// Получение ролей
$roles = db_fetch_all("SELECT * FROM roles ORDER BY name");

// Получение локаций
$locations = db_fetch_all("SELECT * FROM locations WHERE is_active = 1 ORDER BY name");

include '../../layout/header.php';
include '../../layout/sidebar.php';
?>

<main class="main-content">
    <div class="breadcrumbs">
        <a href="<?php echo BASE_URL; ?>">Главная</a> / 
        <a href="index.php">Настройки</a> / 
        Пользователи
    </div>
    
    <div class="page-header">
        <h1>👥 Пользователи</h1>
        <p>Управление учетными записями системы</p>
    </div>
    
    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (ACL::hasPermission($userId, 'settings', 'create')): ?>
    <div class="card">
        <div class="card-header">
            <h2>➕ Добавить пользователя</h2>
        </div>
        <div class="card-body">
            <form method="POST" action="">
                <input type="hidden" name="action" value="add">
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                    <div class="form-group">
                        <label for="login">Логин *</label>
                        <input type="text" id="login" name="login" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="password">Пароль *</label>
                        <input type="password" id="password" name="password" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="full_name">ФИО *</label>
                        <input type="text" id="full_name" name="full_name" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="role_id">Роль *</label>
                        <select id="role_id" name="role_id" class="form-control" required>
                            <option value="">Выберите роль</option>
                            <?php foreach ($roles as $role): ?>
                                <option value="<?php echo $role['id']; ?>"><?php echo htmlspecialchars($role['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Локации доступа</label>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 10px;">
                        <?php foreach ($locations as $location): ?>
                            <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                                <input type="checkbox" name="location_ids[]" value="<?php echo $location['id']; ?>">
                                <span><?php echo htmlspecialchars($location['name']); ?></span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-primary">Добавить пользователя</button>
            </form>
        </div>
    </div>
    <?php endif; ?>
    
    <div class="card">
        <div class="card-header">
            <h2>📋 Список пользователей</h2>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Логин</th>
                            <th>ФИО</th>
                            <th>Роль</th>
                            <th>Локации</th>
                            <th>Статус</th>
                            <th>Дата создания</th>
                            <th>Действия</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                        <tr>
                            <td><?php echo $user['id']; ?></td>
                            <td><strong><?php echo htmlspecialchars($user['login']); ?></strong></td>
                            <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                            <td><?php echo htmlspecialchars($user['role_name']); ?></td>
                            <td><?php echo htmlspecialchars($user['locations'] ?: 'Все'); ?></td>
                            <td>
                                <?php if ($user['is_active']): ?>
                                    <span class="badge badge-success">Активен</span>
                                <?php else: ?>
                                    <span class="badge badge-danger">Неактивен</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('d.m.Y H:i', strtotime($user['created_at'])); ?></td>
                            <td>
                                <?php if ($user['id'] !== $userId): ?>
                                    <?php if (ACL::hasPermission($userId, 'settings', 'update')): ?>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="toggle">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <button type="submit" class="btn btn-secondary" style="padding: 6px 12px; font-size: 12px;">
                                                <?php echo $user['is_active'] ? 'Деактивировать' : 'Активировать'; ?>
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                    
                                    <?php if (ACL::hasPermission($userId, 'settings', 'delete')): ?>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Удалить пользователя?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <button type="submit" class="btn btn-danger" style="padding: 6px 12px; font-size: 12px;">Удалить</button>
                                        </form>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span style="color: var(--text-secondary); font-size: 12px;">Текущий</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</main>

<?php include '../../layout/footer.php'; ?>