<?php
/**
 * Дашборд - главная панель
 * Файл: modules/dashboard/index.php
 */

require_once '../../config.php';
require_once '../../auth/guard.php';
require_once '../../lib/db.php';
require_once '../../lib/acl.php';

$pageTitle = 'Дашборд - ' . APP_NAME;
$userId = $_SESSION['user_id'];

// Получение данных для дашборда
$locationFilter = ACL::getLocationFilter($userId);

// Статистика
$stats = [];

// Общее количество локаций
if (ACL::isAdmin($userId) || ACL::isDirector($userId)) {
    $stats['locations'] = db_fetch_one("SELECT COUNT(*) as count FROM locations WHERE is_active = 1")['count'];
} else {
    $stats['locations'] = count(ACL::getUserLocations($userId));
}

// Общее количество цехов
$workshopsQuery = "SELECT COUNT(*) as count FROM workshops WHERE is_active = 1";
if (!ACL::isAdmin($userId) && !ACL::isDirector($userId)) {
    $locationIds = ACL::getUserLocationIds($userId);
    if (!empty($locationIds)) {
        $locationIdsStr = implode(',', $locationIds);
        $workshopsQuery .= " AND location_id IN ($locationIdsStr)";
    }
}
$stats['workshops'] = db_fetch_one($workshopsQuery)['count'];

// Количество сотрудников
$employeesQuery = "SELECT COUNT(*) as count FROM users WHERE is_active = 1";
$stats['employees'] = db_fetch_one($employeesQuery)['count'];

// Выполненных операций за текущий месяц
$currentMonth = date('Y-m-01');
$operationsQuery = "SELECT COUNT(*) as count FROM productivity_records 
                    WHERE work_date >= '$currentMonth' AND $locationFilter";
$stats['operations'] = db_fetch_one($operationsQuery)['count'];

// Произведено деталей за текущий месяц
$partsQuery = "SELECT COALESCE(SUM(quantity), 0) as total FROM part_production 
               WHERE production_date >= '$currentMonth' AND $locationFilter";
$stats['parts_produced'] = db_fetch_one($partsQuery)['total'];

// Процент брака за текущий месяц
$defectsQuery = "SELECT COALESCE(SUM(quantity), 0) as total FROM defects 
                 WHERE defect_date >= '$currentMonth' AND $locationFilter";
$defectsTotal = db_fetch_one($defectsQuery)['total'];
$stats['defect_rate'] = $stats['parts_produced'] > 0 
    ? round(($defectsTotal / $stats['parts_produced']) * 100, 2) 
    : 0;

// ФОТ за текущий месяц
$wagesQuery = "SELECT COALESCE(SUM(wage_amount), 0) as total FROM productivity_records 
               WHERE work_date >= '$currentMonth' AND $locationFilter";
$stats['total_wages'] = db_fetch_one($wagesQuery)['total'];

// Материалы с низким запасом
$lowStockQuery = "
    SELECT i.name, l.name as location_name, 
           COALESCE(SUM(CASE 
               WHEN im.movement_type IN ('receipt', 'transfer') AND im.to_location_id = l.id THEN im.quantity
               WHEN im.movement_type IN ('deduction', 'transfer') AND im.from_location_id = l.id THEN -im.quantity
               ELSE 0
           END), 0) as current_stock,
           imin.minimum_quantity
    FROM items i
    JOIN item_minimums imin ON imin.item_id = i.id
    JOIN locations l ON l.id = imin.location_id
    LEFT JOIN inventory_movements im ON im.item_id = i.id AND im.item_type = i.item_type
    WHERE i.is_active = 1 AND l.is_active = 1 AND $locationFilter
    GROUP BY i.id, l.id, imin.minimum_quantity
    HAVING current_stock < imin.minimum_quantity
    ORDER BY (current_stock / imin.minimum_quantity) ASC
    LIMIT 10
";
$lowStockItems = db_fetch_all($lowStockQuery);

// Готовая продукция к отгрузке (в цехе упаковки)
$readyProductsQuery = "
    SELECT p.name, l.name as location_name, COUNT(*) as quantity
    FROM product_assembly pa
    JOIN products p ON p.id = pa.product_id
    JOIN locations l ON l.id = pa.location_id
    WHERE $locationFilter
    GROUP BY p.id, l.id
    ORDER BY quantity DESC
    LIMIT 5
";
$readyProducts = db_fetch_all($readyProductsQuery);

// Отгружено изделий за текущий месяц
$shippedQuery = "SELECT COALESCE(SUM(quantity), 0) as total FROM product_shipments 
                 WHERE shipment_date >= '$currentMonth' AND $locationFilter";
$stats['shipped_products'] = db_fetch_one($shippedQuery)['total'];

// План/факт по ключевым деталям за сегодня
$today = date('Y-m-d');
$keyPartsQuery = "
    SELECT p.name, p.daily_plan,
           COALESCE(SUM(pp.quantity), 0) as fact,
           ROUND(COALESCE(SUM(pp.quantity), 0) / p.daily_plan * 100, 1) as percent
    FROM parts p
    LEFT JOIN part_production pp ON pp.part_id = p.id AND pp.production_date = '$today'
    JOIN workshops w ON w.id = p.workshop_id
    WHERE p.is_key = 1 AND p.is_active = 1 AND $locationFilter
    GROUP BY p.id
    ORDER BY percent DESC
    LIMIT 5
";
$keyPartsToday = db_fetch_all($keyPartsQuery);

include '../../layout/header.php';
include '../../layout/sidebar.php';
?>

<main class="main-content">
    <div class="breadcrumbs">
        <a href="<?php echo BASE_URL; ?>">Главная</a> / Дашборд
    </div>
    
    <div class="page-header">
        <h1>Дашборд</h1>
        <p>Обзор ключевых показателей производства</p>
    </div>
    
    <!-- Статистика -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon">🏭</div>
            <div class="stat-info">
                <div class="stat-label">Локации</div>
                <div class="stat-value"><?php echo $stats['locations']; ?></div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">⚙️</div>
            <div class="stat-info">
                <div class="stat-label">Цеха</div>
                <div class="stat-value"><?php echo $stats['workshops']; ?></div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">👥</div>
            <div class="stat-info">
                <div class="stat-label">Сотрудники</div>
                <div class="stat-value"><?php echo $stats['employees']; ?></div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">📊</div>
            <div class="stat-info">
                <div class="stat-label">Операций (месяц)</div>
                <div class="stat-value"><?php echo number_format($stats['operations']); ?></div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">🔧</div>
            <div class="stat-info">
                <div class="stat-label">Деталей произведено</div>
                <div class="stat-value"><?php echo number_format($stats['parts_produced']); ?></div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">❌</div>
            <div class="stat-info">
                <div class="stat-label">Процент брака</div>
                <div class="stat-value"><?php echo $stats['defect_rate']; ?>%</div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">💰</div>
            <div class="stat-info">
                <div class="stat-label">ФОТ (месяц)</div>
                <div class="stat-value"><?php echo number_format($stats['total_wages'], 0, '.', ' '); ?> ₽</div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">📦</div>
            <div class="stat-info">
                <div class="stat-label">Отгружено (месяц)</div>
                <div class="stat-value"><?php echo number_format($stats['shipped_products']); ?></div>
            </div>
        </div>
    </div>
    
    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
        <!-- Материалы с низким запасом -->
        <div class="card">
            <div class="card-header">
                <h2>⚠️ Материалы с низким запасом</h2>
            </div>
            <div class="card-body">
                <?php if (empty($lowStockItems)): ?>
                    <p style="color: #999; text-align: center; padding: 20px;">Все материалы в норме</p>
                <?php else: ?>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>Материал</th>
                                    <th>Локация</th>
                                    <th>Остаток</th>
                                    <th>Минимум</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($lowStockItems as $item): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($item['name']); ?></td>
                                    <td><?php echo htmlspecialchars($item['location_name']); ?></td>
                                    <td><span class="badge badge-danger"><?php echo $item['current_stock']; ?></span></td>
                                    <td><?php echo $item['minimum_quantity']; ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Готовая продукция к отгрузке -->
        <div class="card">
            <div class="card-header">
                <h2>📦 Готовая продукция к отгрузке</h2>
            </div>
            <div class="card-body">
                <?php if (empty($readyProducts)): ?>
                    <p style="color: #999; text-align: center; padding: 20px;">Нет готовой продукции</p>
                <?php else: ?>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>Изделие</th>
                                    <th>Локация</th>
                                    <th>Количество</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($readyProducts as $product): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($product['name']); ?></td>
                                    <td><?php echo htmlspecialchars($product['location_name']); ?></td>
                                    <td><span class="badge badge-success"><?php echo $product['quantity']; ?></span></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- План/факт по ключевым деталям за сегодня -->
    <div class="card">
        <div class="card-header">
            <h2>📈 План/факт по ключевым деталям (сегодня)</h2>
        </div>
        <div class="card-body">
            <?php if (empty($keyPartsToday)): ?>
                <p style="color: #999; text-align: center; padding: 20px;">Нет данных</p>
            <?php else: ?>
                <div class="table-container">
                    <table>
                        <thead>
                            <tr>
                                <th>Деталь</th>
                                <th>План</th>
                                <th>Факт</th>
                                <th>Выполнение</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($keyPartsToday as $part): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($part['name']); ?></td>
                                <td><?php echo $part['daily_plan']; ?></td>
                                <td><?php echo $part['fact']; ?></td>
                                <td>
                                    <span class="badge <?php echo $part['percent'] >= 100 ? 'badge-success' : ($part['percent'] >= 80 ? 'badge-warning' : 'badge-danger'); ?>">
                                        <?php echo $part['percent']; ?>%
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</main>

<script>
// Автообновление данных каждые 60 секунд
setInterval(function() {
    location.reload();
}, 60000);
</script>

<?php include '../../layout/footer.php'; ?>