// assets/app.js
// UI glue for Variant #4:
// - Day/Night theme (localStorage + system default)
// - Floating theme button (since top header is hidden by CSS)
// - Mobile bottom navigation generated from sidebar links

(function(){
  function safe(fn){ try{ fn(); }catch(e){} }

  // Theme init: localStorage > system
  safe(function(){
    const saved = localStorage.getItem('ascrm_theme');
    const prefersDark = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
    const theme = saved ? saved : (prefersDark ? 'dark' : 'light');
    document.documentElement.setAttribute('data-theme', theme);
  });

  function setTheme(next){
    document.documentElement.setAttribute('data-theme', next);
    safe(function(){ localStorage.setItem('ascrm_theme', next); });
  }

  function toggleTheme(){
    const cur = document.documentElement.getAttribute('data-theme') || 'light';
    setTheme(cur === 'dark' ? 'light' : 'dark');
  }

  // Ensure a theme button exists even if header is hidden.
  function ensureThemeFab(){
    if(document.getElementById('themeFab')) return;

    // Reuse existing header button if present
    const headerBtn = document.getElementById('themeToggle');

    const btn = document.createElement('button');
    btn.type = 'button';
    btn.id = 'themeFab';
    btn.className = 'btn theme-fab';
    btn.title = 'Сменить тему';
    btn.innerHTML = '<span class="icon-moon" aria-hidden="true">🌙</span><span class="icon-sun" aria-hidden="true">☀️</span><span class="sr-only">Сменить тему</span>';
    btn.addEventListener('click', function(){ toggleTheme(); });
    document.body.appendChild(btn);

    // Keep header button working too (if it still exists in DOM)
    if(headerBtn){
      headerBtn.addEventListener('click', function(){ toggleTheme(); });
    }
  }

  // Build mobile bottom nav from sidebar items.
  function buildBottomNav(){
    if(document.querySelector('.bottom-nav')) return;

    const sidebar = document.querySelector('.app-sidebar .nav');
    if(!sidebar) return;

    const links = Array.from(sidebar.querySelectorAll('a.nav-item'));
    if(!links.length) return;

    const nav = document.createElement('nav');
    nav.className = 'bottom-nav';
    nav.setAttribute('aria-label', 'Навигация');

    // We keep it template-like (row of icon buttons). If there are many modules,
    // CSS grid will still fit; on very small screens it will shrink. Prefer icons.
    links.forEach(function(a){
      const b = document.createElement('a');
      b.href = a.getAttribute('href') || '#';
      b.className = 'btn' + (a.classList.contains('active') ? ' active' : '');
      const ico = a.querySelector('.nav-ico');
      const title = a.querySelector('.nav-title');
      const icoText = ico ? ico.textContent : '•';
      const label = title ? title.textContent : 'Раздел';
      b.innerHTML = '<span aria-hidden="true">' + escapeHtml(icoText.trim()) + '</span><span class="sr-only">' + escapeHtml(label) + '</span>';
      nav.appendChild(b);
    });

    document.body.appendChild(nav);
  }

  function escapeHtml(s){
    return String(s)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  // Init after DOM ready
  document.addEventListener('DOMContentLoaded', function(){
    ensureThemeFab();
    buildBottomNav();

    // Optional: dashboard feed auto refresh (kept from baseline app.js)
    const feed = document.getElementById('feed');
    if(feed){
      setInterval(() => {
        const now = new Date();
        const el = document.createElement('div');
        el.className = 'feed-item';
        el.innerHTML = `
          <div class="dot warn"></div>
          <div>
            <div class="feed-title">Автообновление</div>
            <div class="feed-sub muted">Время: ${now.toLocaleTimeString()}</div>
          </div>
          <div class="feed-time muted">AJAX</div>
        `;
        feed.prepend(el);
        while(feed.children.length > 5) feed.removeChild(feed.lastElementChild);
      }, 60000);
    }
  });
})();
