<?php
// app/warehouse.php — общие функции склада
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/audit.php';
require_once __DIR__ . '/notify.php';

function wh_get_user_locations_for_warehouse($user) {
    // admin/director — все локации; остальные — только назначенные
    if (in_array($user['role'], ['admin','director'], true)) {
        return db_fetch_all(db_query("SELECT id AS location_id, name AS location_name FROM locations ORDER BY name"));
    }
    return get_user_locations((int)$user['id']); // из auth.php
}

function wh_require_location_access_by_user($user, $location_id) {
    if (in_array($user['role'], ['admin','director'], true)) return;
    require_location_access((int)$user['id'], (int)$location_id); // auth.php
}

function wh_get_place_id($location_id, $type, $workshop_id = 0) {
    $location_id = (int)$location_id;
    $type_esc = db_escape($type);
    if ($type === 'warehouse') {
        $row = db_fetch_one(db_query("SELECT id FROM storage_places WHERE location_id=$location_id AND place_type='warehouse' LIMIT 1"));
        return $row ? (int)$row['id'] : 0;
    }
    if ($type === 'workshop') {
        $workshop_id = (int)$workshop_id;
        $row = db_fetch_one(db_query("SELECT id FROM storage_places WHERE place_type='workshop' AND workshop_id=$workshop_id LIMIT 1"));
        return $row ? (int)$row['id'] : 0;
    }
    return 0;
}

function wh_ensure_places_exist() {
    // создаём склады и цеха, если кто-то добавил локацию/цех после миграции
    db_query("INSERT INTO storage_places (location_id, place_type, name, created_at)
              SELECT l.id, 'warehouse', CONCAT('Склад: ', l.name), NOW()
              FROM locations l
              LEFT JOIN storage_places sp ON sp.location_id=l.id AND sp.place_type='warehouse'
              WHERE sp.id IS NULL");

    db_query("INSERT INTO storage_places (location_id, place_type, workshop_id, name, created_at)
              SELECT w.location_id, 'workshop', w.id, CONCAT('Цех: ', w.name), NOW()
              FROM workshops w
              LEFT JOIN storage_places sp ON sp.workshop_id=w.id AND sp.place_type='workshop'
              WHERE sp.id IS NULL");
}

function wh_get_balance($item_type, $item_id, $place_id) {
    $item_type_esc = db_escape($item_type);
    $item_id = (int)$item_id;
    $place_id = (int)$place_id;

    $sql = "
      SELECT COALESCE(SUM(delta),0) AS bal FROM (
        SELECT qty AS delta FROM stock_moves 
          WHERE item_type='$item_type_esc' AND item_id=$item_id AND to_place_id=$place_id AND is_reversed=0
        UNION ALL
        SELECT -qty AS delta FROM stock_moves 
          WHERE item_type='$item_type_esc' AND item_id=$item_id AND from_place_id=$place_id AND is_reversed=0
      ) x";
    $row = db_fetch_one(db_query($sql));
    return $row ? (float)$row['bal'] : 0.0;
}

function wh_get_balances_by_place($item_type, $item_id, $location_id) {
    $item_type_esc = db_escape($item_type);
    $item_id = (int)$item_id;
    $location_id = (int)$location_id;

    $sql = "
      SELECT p.id AS place_id, p.name, p.place_type,
             COALESCE(SUM(t.in_qty),0) - COALESCE(SUM(t.out_qty),0) AS qty
      FROM storage_places p
      LEFT JOIN (
        SELECT to_place_id AS place_id, SUM(qty) AS in_qty, 0 AS out_qty
          FROM stock_moves
         WHERE item_type='$item_type_esc' AND item_id=$item_id AND is_reversed=0
         GROUP BY to_place_id
        UNION ALL
        SELECT from_place_id AS place_id, 0 AS in_qty, SUM(qty) AS out_qty
          FROM stock_moves
         WHERE item_type='$item_type_esc' AND item_id=$item_id AND is_reversed=0
         GROUP BY from_place_id
      ) t ON t.place_id=p.id
      WHERE p.location_id=$location_id AND p.is_active=1
      GROUP BY p.id, p.name, p.place_type
      HAVING qty <> 0
      ORDER BY FIELD(p.place_type,'warehouse','workshop','employee','virtual_parts'), p.name";
    return db_fetch_all(db_query($sql));
}

function wh_list_places_for_location($location_id) {
    $location_id = (int)$location_id;
    return db_fetch_all(db_query("SELECT * FROM storage_places WHERE location_id=$location_id AND is_active=1 ORDER BY FIELD(place_type,'warehouse','workshop','employee','virtual_parts'), name"));
}

function wh_move_create($data, $user_id) {
    // data: item_type,item_id,move_type,from_place_id,to_place_id,qty,unit_price,receipt_id,reason,comment,reversed_of
    $it = db_escape($data['item_type']);
    $iid = (int)$data['item_id'];
    $mt = db_escape($data['move_type']);
    $from = isset($data['from_place_id']) ? (int)$data['from_place_id'] : 'NULL';
    $to = isset($data['to_place_id']) ? (int)$data['to_place_id'] : 'NULL';
    $qty = (float)$data['qty'];
    $qty_sql = (string)$qty;

    $unit_price_sql = 'NULL';
    if (isset($data['unit_price']) && $data['unit_price'] !== '' && $data['unit_price'] !== null) {
        $p = str_replace(',', '.', (string)$data['unit_price']);
        if (is_numeric($p)) $unit_price_sql = (string)((float)$p);
    }
    $receipt_id_sql = isset($data['receipt_id']) && (int)$data['receipt_id']>0 ? (int)$data['receipt_id'] : 'NULL';
    $reason = db_escape((string)($data['reason'] ?? ''));
    $comment = db_escape((string)($data['comment'] ?? ''));
    $reversed_of_sql = isset($data['reversed_of']) && (int)$data['reversed_of']>0 ? (int)$data['reversed_of'] : 'NULL';

    db_query("INSERT INTO stock_moves(item_type,item_id,move_type,from_place_id,to_place_id,qty,unit_price,receipt_id,reason,comment,reversed_of,created_at,created_by)
              VALUES ('$it',$iid,'$mt',".($from==='NULL'?'NULL':$from).",".($to==='NULL'?'NULL':$to).",$qty_sql,$unit_price_sql,$receipt_id_sql,'$reason','$comment',$reversed_of_sql,NOW(),".(int)$user_id.")");
    $id = (int)db_last_id();
    audit('CREATE','stock_moves',$id,(int)$user_id, ['item_type'=>$data['item_type'],'item_id'=>$iid,'move_type'=>$data['move_type'],'qty'=>$qty]);
    // Telegram: проверка минимума (только если меняется общий остаток по локации)
if (in_array($data['move_type'], ['writeoff','transfer','reverse','adjust'], true) && function_exists('notify_low_stock_check')) {
    $loc_from = 0;
    $loc_to = 0;
    if (!empty($data['from_place_id'])) {
        $p = db_fetch_one(db_query("SELECT location_id FROM storage_places WHERE id=".(int)$data['from_place_id']." LIMIT 1"));
        if ($p) $loc_from = (int)$p['location_id'];
    }
    if (!empty($data['to_place_id'])) {
        $p = db_fetch_one(db_query("SELECT location_id FROM storage_places WHERE id=".(int)$data['to_place_id']." LIMIT 1"));
        if ($p) $loc_to = (int)$p['location_id'];
    }
    if ($loc_from>0) notify_low_stock_check($data['item_type'], $iid, $loc_from);
    if ($loc_to>0 && $loc_to!==$loc_from) notify_low_stock_check($data['item_type'], $iid, $loc_to);
}

    return $id;
}


function wh_get_location_total_balance($item_type, $item_id, $location_id) {
    $item_type_esc = db_escape($item_type);
    $item_id = (int)$item_id;
    $location_id = (int)$location_id;

    // Сумма по всем местам хранения внутри локации
    $row = db_fetch_one(db_query("
      SELECT COALESCE(SUM(delta),0) AS bal FROM (
        SELECT sm.qty AS delta
        FROM stock_moves sm
        JOIN storage_places p ON p.id=sm.to_place_id
        WHERE sm.item_type='$item_type_esc' AND sm.item_id=$item_id AND sm.is_reversed=0 AND p.location_id=$location_id
        UNION ALL
        SELECT -sm.qty AS delta
        FROM stock_moves sm
        JOIN storage_places p ON p.id=sm.from_place_id
        WHERE sm.item_type='$item_type_esc' AND sm.item_id=$item_id AND sm.is_reversed=0 AND p.location_id=$location_id
      ) x
    "));
    return $row ? (float)$row['bal'] : 0.0;
}
