-- db/update_002_warehouse.sql
-- Склад: места хранения + журнал движений + приходные документы
SET NAMES utf8mb4;

CREATE TABLE IF NOT EXISTS storage_places (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  location_id INT UNSIGNED NOT NULL,
  place_type ENUM('warehouse','workshop','employee','virtual_parts') NOT NULL,
  workshop_id INT UNSIGNED DEFAULT NULL,
  employee_id INT UNSIGNED DEFAULT NULL,
  name VARCHAR(255) NOT NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL,
  PRIMARY KEY (id),
  KEY idx_place_loc (location_id),
  KEY idx_place_type (place_type),
  UNIQUE KEY uq_place_wh (location_id, place_type),
  CONSTRAINT fk_place_loc FOREIGN KEY (location_id) REFERENCES locations(id) ON DELETE CASCADE,
  CONSTRAINT fk_place_ws FOREIGN KEY (workshop_id) REFERENCES workshops(id) ON DELETE SET NULL
  -- employee_id добавим позже, когда появится таблица персонала
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Документы прихода (поставщик/накладная/комментарий)
CREATE TABLE IF NOT EXISTS purchase_receipts (
  id INT UNSIGNED NOT NULL AUTO_INCREMENT,
  location_id INT UNSIGNED NOT NULL,
  supplier VARCHAR(255) NOT NULL DEFAULT '',
  invoice_no VARCHAR(128) NOT NULL DEFAULT '',
  comment TEXT,
  created_at DATETIME NOT NULL,
  created_by INT UNSIGNED NOT NULL,
  PRIMARY KEY (id),
  KEY idx_pr_loc (location_id),
  KEY idx_pr_dt (created_at),
  CONSTRAINT fk_pr_loc FOREIGN KEY (location_id) REFERENCES locations(id) ON DELETE CASCADE,
  CONSTRAINT fk_pr_user FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Журнал движений
CREATE TABLE IF NOT EXISTS stock_moves (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  item_type ENUM('material','tool','consumable') NOT NULL,
  item_id INT UNSIGNED NOT NULL,
  move_type ENUM('receipt','transfer','writeoff','adjust','reverse') NOT NULL,
  from_place_id INT UNSIGNED DEFAULT NULL,
  to_place_id INT UNSIGNED DEFAULT NULL,
  qty DECIMAL(14,3) NOT NULL DEFAULT 0,
  unit_price DECIMAL(12,2) DEFAULT NULL, -- цена партии (для прихода)
  receipt_id INT UNSIGNED DEFAULT NULL,
  reason VARCHAR(255) NOT NULL DEFAULT '',
  comment TEXT,
  is_reversed TINYINT(1) NOT NULL DEFAULT 0,
  reversed_of BIGINT UNSIGNED DEFAULT NULL,
  created_at DATETIME NOT NULL,
  created_by INT UNSIGNED NOT NULL,
  PRIMARY KEY (id),
  KEY idx_sm_item (item_type, item_id),
  KEY idx_sm_dt (created_at),
  KEY idx_sm_from (from_place_id),
  KEY idx_sm_to (to_place_id),
  CONSTRAINT fk_sm_from FOREIGN KEY (from_place_id) REFERENCES storage_places(id) ON DELETE SET NULL,
  CONSTRAINT fk_sm_to FOREIGN KEY (to_place_id) REFERENCES storage_places(id) ON DELETE SET NULL,
  CONSTRAINT fk_sm_pr FOREIGN KEY (receipt_id) REFERENCES purchase_receipts(id) ON DELETE SET NULL,
  CONSTRAINT fk_sm_user FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Автосоздание "Склад" для каждой локации (если отсутствует)
INSERT INTO storage_places (location_id, place_type, name, created_at)
SELECT l.id, 'warehouse', CONCAT('Склад: ', l.name), NOW()
FROM locations l
LEFT JOIN storage_places sp ON sp.location_id=l.id AND sp.place_type='warehouse'
WHERE sp.id IS NULL;

-- Автосоздание мест хранения типа "Цех" для существующих цехов
INSERT INTO storage_places (location_id, place_type, workshop_id, name, created_at)
SELECT w.location_id, 'workshop', w.id, CONCAT('Цех: ', w.name), NOW()
FROM workshops w
LEFT JOIN storage_places sp ON sp.workshop_id=w.id AND sp.place_type='workshop'
WHERE sp.id IS NULL;
