<?php
// app/xlsx.php — минимальный генератор .xlsx без сторонних библиотек
// Подходит для простых таблиц: строки/колонки с текстом и числами.
// Важно: не используем PDO, только стандартные функции PHP.

function xlsx_escape($s) {
    $s = (string)$s;
    $s = str_replace(['&','<','>','"',"\'"], ['&amp;','&lt;','&gt;','&quot;','&apos;'], $s);
    return $s;
}

function xlsx_col_name($n) { // 1->A
    $name = '';
    while ($n > 0) {
        $rem = ($n - 1) % 26;
        $name = chr(65 + $rem) . $name;
        $n = (int)(($n - 1) / 26);
    }
    return $name;
}

function xlsx_build_sheet_xml($rows) {
    $xml = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>';
    $xml .= '<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">';
    $xml .= '<sheetData>';

    $rnum = 1;
    foreach ($rows as $row) {
        $xml .= '<row r="'.$rnum.'">';
        $cnum = 1;
        foreach ($row as $cell) {
            $ref = xlsx_col_name($cnum) . $rnum;

            // число?
            $is_num = is_int($cell) || is_float($cell) || (is_string($cell) && preg_match('/^-?\d+(\.\d+)?$/', $cell));
            if ($is_num) {
                $val = (string)$cell;
                $xml .= '<c r="'.$ref.'"><v>'.xlsx_escape($val).'</v></c>';
            } else {
                $val = xlsx_escape($cell);
                $xml .= '<c r="'.$ref.'" t="inlineStr"><is><t>'.$val.'</t></is></c>';
            }
            $cnum++;
        }
        $xml .= '</row>';
        $rnum++;
    }

    $xml .= '</sheetData></worksheet>';
    return $xml;
}

function xlsx_output($filename, $rows, $sheet_name='Sheet1') {
    // создаём xlsx как zip
    $tmp = tempnam(sys_get_temp_dir(), 'xlsx');
    $zip = new ZipArchive();
    $zip->open($tmp, ZipArchive::OVERWRITE);

    $content_types = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
      .'<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">'
      .'<Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>'
      .'<Default Extension="xml" ContentType="application/xml"/>'
      .'<Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>'
      .'<Override PartName="/xl/worksheets/sheet1.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>'
      .'</Types>';

    $rels = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
      .'<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">'
      .'<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>'
      .'</Relationships>';

    $workbook = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
      .'<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">'
      .'<sheets><sheet name="'.xlsx_escape($sheet_name).'" sheetId="1" r:id="rId1"/></sheets>'
      .'</workbook>';

    $workbook_rels = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>'
      .'<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">'
      .'<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/sheet1.xml"/>'
      .'</Relationships>';

    $sheet = xlsx_build_sheet_xml($rows);

    $zip->addFromString('[Content_Types].xml', $content_types);
    $zip->addFromString('_rels/.rels', $rels);
    $zip->addFromString('xl/workbook.xml', $workbook);
    $zip->addFromString('xl/_rels/workbook.xml.rels', $workbook_rels);
    $zip->addFromString('xl/worksheets/sheet1.xml', $sheet);

    $zip->close();

    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="'.$filename.'"');
    header('Content-Length: '.filesize($tmp));
    readfile($tmp);
    @unlink($tmp);
    exit;
}
